<?php
/**
 * Plugin Name: Seamless WebP
 * Plugin URI: https://byburk.net
 * Description: Automatically converts images to WebP and seamlessly serves them without changing URLs.
 * Version: 1.2.0
 * Author: Burk
 * License: GPL v2 or later
 * Text Domain: seamless-webp
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

class SeamlessWebP {
    
    public function __construct() {
        // Convert images on upload (after sizes generated)

        add_filter('wp_generate_attachment_metadata', array($this, 'handle_metadata_hook'), 10, 2);
        
        // Admin Assets
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // Add settings link

        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));
        
        // Admin notice
        add_action('admin_notices', array($this, 'admin_notice'));
        
        // Bulk Convert Menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // AJAX Handler
        add_action('wp_ajax_seamless_webp_bulk_process', array($this, 'ajax_bulk_process'));
    }

    /**
     * Enqueue Admin Assets
     */
    public function enqueue_admin_assets($hook) {
        if ($hook !== 'media_page_seamless-webp') {
            return;
        }

        wp_enqueue_script(
            'seamless-webp-admin',
            plugin_dir_url(__FILE__) . 'assets/js/admin.js',
            array('jquery'),
            '1.2.0',
            true
        );

        // Stats for localization
        $query_images_args = array(
            'post_type'      => 'attachment',
            'post_mime_type' => array('image/jpeg', 'image/png'),
            'post_status'    => 'inherit',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        );
        $query_images = new WP_Query($query_images_args);
        $total_count = $query_images->found_posts;

        wp_localize_script('seamless-webp-admin', 'seamless_webp_vars', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seamless_webp_nonce'),
            'total_count' => $total_count,
            'strings' => array(
                'starting' => __('Starting conversion...', 'seamless-webp'),
                'running_test' => __('Running test batch...', 'seamless-webp'),
                'test_complete' => __('Test Complete! 5 images checked/converted.', 'seamless-webp'),
                'all_done' => __('All images processed! 🎉', 'seamless-webp'),
                'server_error' => __('Server error occurred. Please refresh and try again.', 'seamless-webp'),
                'enabling' => __('Enabling...', 'seamless-webp'),
                'enable_btn' => __('Enable WebP Delivery', 'seamless-webp'),
                'refresh_hint' => __('Refesh page to see updates.', 'seamless-webp'),
            )
        ));
    }

    /**
     * Hook into metadata generation (runs after all sizes are created)
     */
    public function handle_metadata_hook($metadata, $attachment_id) {
        $file_path = get_attached_file($attachment_id);
        
        // Convert main file
        if ($file_path && file_exists($file_path)) {
            // Check mime type (simple check via extension as mime type might be tricky here)
            if (preg_match('/\.(jpe?g|png)$/i', $file_path)) {
                $this->convert_file($file_path);
                
                // Convert sub-sizes
                if (!empty($metadata['sizes']) && is_array($metadata['sizes'])) {
                    $dir_path = dirname($file_path);
                    foreach ($metadata['sizes'] as $size_info) {
                        if (isset($size_info['file'])) {
                            $sub_file_path = $dir_path . '/' . $size_info['file'];
                            if (file_exists($sub_file_path)) {
                                $this->convert_file($sub_file_path);
                            }
                        }
                    }
                }
            }
        }
        
        return $metadata;
    }


    /**
     * Core logic: Convert a single file to WebP
     * Returns array with success status and stats, or false on failure
     */
    private function convert_file($file_path) {
        $webp_path = preg_replace('/\.(jpe?g|png)$/i', '.webp', $file_path);
        
        // Skip if WebP already exists (saves time in bulk process)
        if (file_exists($webp_path)) {
            return array('status' => 'skipped', 'msg' => 'Already exists');
        }

        $image_type = exif_imagetype($file_path);
        $image = null;

        switch ($image_type) {
            case IMAGETYPE_JPEG:
                $image = @imagecreatefromjpeg($file_path);
                break;
            case IMAGETYPE_PNG:
                $image = @imagecreatefrompng($file_path);
                if ($image) {
                    imagepalettetotruecolor($image);
                    imagealphablending($image, true);
                    imagesavealpha($image, true);
                }
                break;
        }
        
        if (!$image) {
            return array('status' => 'error', 'msg' => 'Could not load image');
        }
        
        // Convert to WebP (85% quality)
        $success = imagewebp($image, $webp_path, 85);
        imagedestroy($image);
        
        if ($success) {
            $original_size = filesize($file_path);
            $webp_size = filesize($webp_path);
            $savings = ($original_size > 0) ? round((($original_size - $webp_size) / $original_size) * 100, 1) : 0;
            
            $savings = ($original_size > 0) ? round((($original_size - $webp_size) / $original_size) * 100, 1) : 0;
            
            return array('status' => 'success', 'savings' => $savings);
        }
        
        return array('status' => 'error', 'msg' => 'Conversion failed');
    }
    
    /**
     * Add menu item
     */
    public function add_admin_menu() {
        add_submenu_page(
            'upload.php',
            'Seamless WebP',
            'Seamless WebP',
            'manage_options',
            'seamless-webp',
            array($this, 'render_admin_page')
        );
    }
    
    /**
     * Render the Admin Page
     */
    public function render_admin_page() {
        // Calculate stats (logic moved to enqueue, but need counts for initial render or just use 0 and let JS update? 
        // Better to allow JS to handle it or reuse the query if we want serverside render of initial stats.
        // For simplicity, we can't easily share the $total_count from enqueue without global or re-query.
        // Re-querying is fine for admin page load.
        $query_images_args = array(
            'post_type'      => 'attachment',
            'post_mime_type' => array('image/jpeg', 'image/png'),
            'post_status'    => 'inherit',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        );
        $query_images = new WP_Query($query_images_args);
        $total_count = $query_images->found_posts;

        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Seamless WebP', 'seamless-webp'); ?></h1>
            <div class="card" style="max-width: 600px; margin-top: 20px; padding: 20px;">
                <h2><?php echo esc_html__('Bulk Convert Library', 'seamless-webp'); ?></h2>
                <p><?php echo esc_html__('This tool will check all your JPG and PNG images and generate WebP versions if they are missing.', 'seamless-webp'); ?></p>
                
                <div style="margin: 20px 0; padding: 15px; background: #f0f6fc; border-radius: 4px;">
                    <strong><?php echo esc_html__('Stats:', 'seamless-webp'); ?></strong><br>
                    <?php echo esc_html__('Total Images found:', 'seamless-webp'); ?> <strong><?php echo intval($total_count); ?></strong>
                </div>

                <div id="seamless-webp-progress-wrap" style="display:none; margin-bottom: 20px;">
                    <div style="background: #ddd; border-radius: 10px; overflow: hidden; height: 20px;">
                        <div id="seamless-webp-progress-bar" style="width: 0%; background: #2271b1; height: 100%; transition: width 0.3s;"></div>
                    </div>
                    <p style="text-align: center; margin-top: 5px;">
                        <?php echo esc_html__('Processed:', 'seamless-webp'); ?> <span id="seamless-webp-count">0</span> / <?php echo intval($total_count); ?>
                    </p>
                </div>

                <div style="display: flex; gap: 10px; align-items: center;">
                    <button id="seamless-webp-test-btn" class="button button-secondary"><?php echo esc_html__('Test Convert (5 Images)', 'seamless-webp'); ?></button>
                    <button id="seamless-webp-start-btn" class="button button-primary button-hero"><?php echo esc_html__('Start Bulk Conversion', 'seamless-webp'); ?></button>
                    <span id="seamless-webp-spinner" class="spinner" style="float:none; margin:0;"></span>
                </div>
                
                <div id="seamless-webp-msg" style="margin-top: 10px; font-weight: bold;"></div>
            </div>

            <!-- WebP Delivery Settings -->
            <div class="card" style="max-width: 600px; margin-top: 20px; padding: 20px;">
                <h2><?php echo esc_html__('WebP Delivery', 'seamless-webp'); ?></h2>
                <?php
                $htaccess_path = ABSPATH . '.htaccess';
                $is_active = $this->check_htaccess_rules($htaccess_path);
                ?>
                <div style="display: flex; align-items: center; justify-content: space-between;">
                    <div>
                        <strong><?php echo esc_html__('Status:', 'seamless-webp'); ?> </strong>
                        <?php if ($is_active): ?>
                            <span style="color: #46b450; font-weight: bold;"><?php echo esc_html__('Active ✅', 'seamless-webp'); ?></span>
                            <p class="description"><?php echo esc_html__('Your server is configured to serve WebP files.', 'seamless-webp'); ?></p>
                        <?php else: ?>
                            <span style="color: #dc3232; font-weight: bold;"><?php echo esc_html__('Inactive ❌', 'seamless-webp'); ?></span>
                            <p class="description"><?php echo esc_html__('WebP files will be created but not served to visitors.', 'seamless-webp'); ?></p>
                        <?php endif; ?>
                    </div>
                    <?php if (!$is_active): ?>
                        <button id="seamless-webp-enable-btn" class="button button-primary"><?php echo esc_html__('Enable WebP Delivery', 'seamless-webp'); ?></button>
                    <?php else: ?>
                        <button class="button button-secondary" disabled><?php echo esc_html__('Enabled', 'seamless-webp'); ?></button>
                    <?php endif; ?>
                </div>
                <div id="seamless-webp-htaccess-msg" style="margin-top: 10px;"></div>
            </div>
        </div>

        <?php
    }

    /**
     * AJAX Handler for Bulk Process
     */
    public function ajax_bulk_process() {
        // Check permissions and nonce (simple check)
        if (!current_user_can('manage_options') || !check_ajax_referer('seamless_webp_nonce', 'nonce', false)) {
            wp_send_json_error('Permission denied');
        }

        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : 5;

        // Get batch of images
        $args = array(
            'post_type'      => 'attachment',
            'post_mime_type' => array('image/jpeg', 'image/png'),
            'post_status'    => 'inherit',
            'posts_per_page' => $batch_size,
            'offset'         => $offset,
            'fields'         => 'ids',
            'orderby'        => 'ID',
            'order'          => 'ASC'
        );
        
        $query = new WP_Query($args);
        $posts = $query->posts;
        $count = 0;

        foreach ($posts as $post_id) {
            // Convert main file
            $file_path = get_attached_file($post_id);
            if ($file_path && file_exists($file_path)) {
                $this->convert_file($file_path);
                $count++; // Count main image
            }

            // Convert all generated sizes
            $metadata = wp_get_attachment_metadata($post_id);
            if (!empty($metadata['sizes']) && is_array($metadata['sizes'])) {
                $dir_path = dirname($file_path);
                foreach ($metadata['sizes'] as $size_info) {
                    if (isset($size_info['file'])) {
                        $sub_file_path = $dir_path . '/' . $size_info['file'];
                        if (file_exists($sub_file_path)) {
                            $this->convert_file($sub_file_path);
                             // We don't increment count per sub-size to keep progress bar consistent with total attachments
                        }
                    }
                }
            }
        }

        wp_send_json_success(array('count' => $count));
    }
    
    /**
     * Add settings link to plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('upload.php?page=seamless-webp') . '">' . __('Settings', 'seamless-webp') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Helper to check if rules exist
     */
    private function check_htaccess_rules($path) {
        if (!file_exists($path)) return false;
        $content = file_get_contents($path);
        // Check for our specific marker
        return (strpos($content, '# BEGIN SeamlessWebP') !== false);
    }

    /**
     * AJAX Handler to write .htaccess
     */
    public function ajax_save_htaccess() {
        if (!current_user_can('manage_options') || !check_ajax_referer('seamless_webp_nonce', 'nonce', false)) {
            wp_send_json_error(__('Permission denied', 'seamless-webp'));
        }

        $htaccess_path = ABSPATH . '.htaccess';
        
        if (!file_exists($htaccess_path)) {
            // Create if likely manageable
            if (!wp_is_writable(ABSPATH)) {
               wp_send_json_error(__('.htaccess does not exist and root is not writable.', 'seamless-webp'));
            }
            // touch($htaccess_path); // Removed to avoid direct filesystem call warning, insert_with_markers often creates if missing, or we rely on writable dir
        }

        if (file_exists($htaccess_path) && !wp_is_writable($htaccess_path)) {
            wp_send_json_error(__('.htaccess is not writable.', 'seamless-webp'));
        }

        // Just blindly try to update/insert, insert_with_markers handles existence check nicely typically,
        // but since we want to toggle or set, let's just set.
        
        $rules = array(
                 "<IfModule mod_rewrite.c>",
                 "    RewriteEngine On",
                 "    RewriteCond %{HTTP_ACCEPT} image/webp",
                 "    RewriteCond %{REQUEST_FILENAME} (.*)\.(jpe?g|png)$",
                 "    RewriteCond %1.webp -f",
                 "    RewriteRule ^(.+)\.(jpe?g|png)$ $1.webp [T=image/webp,E=accept:1,L]",
                 "</IfModule>",
                 "<IfModule mod_headers.c>",
                 "    Header append Vary Accept env=REDIRECT_accept",
                 "</IfModule>",
                 "AddType image/webp .webp"
        );

        // Ensure insert_with_markers is available
        if (!function_exists('insert_with_markers')) {
            require_once ABSPATH . 'wp-admin/includes/misc.php';
        }

        if (insert_with_markers($htaccess_path, 'SeamlessWebP', $rules)) {
            wp_send_json_success(__('Rules added successfully.', 'seamless-webp'));
        } else {
            wp_send_json_error(__('Failed to write to file.', 'seamless-webp'));
        }
    }

    /**
     * Admin notice with instructions (Legacy or if write failed)
     */
    public function admin_notice() {
        $screen = get_current_screen();
        if ($screen->id !== 'plugins' && $screen->id !== 'media_page_seamless-webp') {
            return;
        }
        
        // Check if .htaccess has WebP rules
        $htaccess_path = ABSPATH . '.htaccess';
        $has_webp_rules = false;
        
        if (file_exists($htaccess_path)) {
            $has_webp_rules = $this->check_htaccess_rules($htaccess_path);
        }
        
        if (!$has_webp_rules) {
            ?>
            <div class="notice notice-warning">
                <p><strong>Seamless WebP:</strong> To serve WebP images, add this to your <code>.htaccess</code>:</p>
                <textarea style="width: 100%; height: 150px; font-family: monospace; background: #f0f0f0;">
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteCond %{HTTP_ACCEPT} image/webp
    RewriteCond %{REQUEST_FILENAME} (.*)\.(jpe?g|png)$
    RewriteCond %1.webp -f
    RewriteRule ^(.+)\.(jpe?g|png)$ $1.webp [T=image/webp,E=accept:1,L]
</IfModule>
<IfModule mod_headers.c>
    Header append Vary Accept env=REDIRECT_accept
</IfModule>
AddType image/webp .webp
                </textarea>
            </div>
            <?php
        }
    }

    /**
     * Deactivation Hook
     */
    public static function on_deactivation() {
        if (!current_user_can('activate_plugins')) return;
        
        $htaccess_path = ABSPATH . '.htaccess';
        if (!file_exists($htaccess_path) || !wp_is_writable($htaccess_path)) return;

        if (!function_exists('insert_with_markers')) {
            require_once ABSPATH . 'wp-admin/includes/misc.php';
        }
        
        // Remove rules
        insert_with_markers($htaccess_path, 'SeamlessWebP', array());
    }
}

// Initialize plugin
new SeamlessWebP();
register_deactivation_hook(__FILE__, array('SeamlessWebP', 'on_deactivation'));
